function spring_mass_walking_grf
%% Tabula Rasa
% clear all
% close all
% clc

%% Parameters

l0 = 1;             %uncompressed spring length [m]
m = 80;             %mass [kg]
k = 23000;          %spring stiffness [N/m]
g = 9.81;           %acceleration due to gravity [m/s^2]
alpha0 = 70;%75;	%landing angle of leg [deg]

alpha0 = deg2rad(alpha0);

n_steps = 50;

%% Initial Conditions

x = 0;
y = 1;
x_dot = 1.3; %1.3;
y_dot = 0;

% x1 = -l0*cos(alpha0);
x0 = 0;

%% State variable
traj = [];
traj_temp =[];
q = [x y x_dot y_dot]';

%% Transition Conditions

foot_impact = odeset('Events', @double_stance);
push_off = odeset('Events', @single_stance);
peak = odeset('Events', @apex);

%% Simulation
for i = 1:n_steps
    [t,q] = ode45(@EoM_single, [0,5], q,foot_impact);
    step_traj = q;
    traj_temp = [traj_temp; q];
    q = q(end,:);
    x1 = q(1) + l0*cos(alpha0);
    if(i==n_steps)
        q_single = [q_single;step_traj];
        t_single = [t_single;t+t_single(end)];
    end
    
    [t,q] = ode45(@EoM_double, [0,5], q,push_off);
    step_traj = q;
    traj_temp = [traj_temp; q];
    traj_temp(:,1) = traj_temp(:,1) + x0;
    traj = [traj; traj_temp];
    x0 = x0 + x1;
    q = [traj(end,1) - x0 traj(end,2:4)]';
    if(i==n_steps-1)
        q_double1 = step_traj;
        t_double1 = t;
    end
    if(i==n_steps)
        q_double2 = step_traj;
        t_double2 = t;
    end
    
    [t,q] = ode45(@EoM_single, [0,5], q,peak);
    traj_temp = q;
    q = q(end,:);
    if(i==n_steps-1)
        q_single = traj_temp;
        t_single = t;
    end
end
traj_temp(:,1) = traj_temp(:,1) + x0;
traj = [traj; traj_temp];

%% Plot
h = figure(41);
plot(traj(:,1),traj(:,2));
title(['Trajectory of ' num2str(n_steps) ' steps for spring mass walking'])
xlabel('x (m)')
ylabel('y (m)')

%% Output Figure - uncomment to print a .eps file.
% print sprint_mass_walking_traj.eps -depsc2

%% Ground reaction force
% Double Stance 1
pos_double = q_double1;
x1 = pos_double(1,1) + l0*cos(alpha0);

for i = 1:length(pos_double)
    % Calculate leg length and angle
    l1 = sqrt((x1-pos_double(i,1))^2+pos_double(i,2)^2);
    F1 = k*(l0-l1);
    alpha1 = atan(pos_double(i,2)/(x1-pos_double(i,1)));

    % Forces from spring equation
    Fx_double_1(i) = -cos(alpha1)*F1;
    Fy_double_1(i) = sin(alpha1)*F1;
end

% Double Stance 2
pos_double = q_double2;
x1 = pos_double(1,1) + l0*cos(alpha0);
for i = 1:length(pos_double)
    % Calculate leg length and angle
    l2 = sqrt(pos_double(i,1)^2+pos_double(i,2)^2);
    F2 = k*(l0-l2);
    alpha2 = atan(pos_double(i,2)/pos_double(i,1));
    
    % Forces from spring equation
    Fx_double_2(i) = cos(alpha2)*F2;
    Fy_double_2(i) = sin(alpha2)*F2;
end

% Single stance grf from accelerations
for i = 1:length(q_single)
    q_dot_temp = EoM_single(1,q_single(i,:));
    pos_ddot_single(i,1:2) = q_dot_temp(3:4)';
end

Fx_single = m*pos_ddot_single(:,1);
Fy_single = m*pos_ddot_single(:,2) + m*g;

% Put pieces together & normalize
Fx = [Fx_double_1';Fx_single;Fx_double_2'];
Fy = [Fy_double_1';Fy_single;Fy_double_2'];
Fx = Fx/(m*g);
Fy = Fy/(m*g);

t_all = [t_double1; t_single+t_double1(end);t_double2+t_single(end)+t_double1(end)];
t_all = t_all/t_all(end);


% Plot grf
figure(42);
clf
hold on
plot(t_all,Fy,'k');
plot(t_all,Fx,'b');
xlabel('time [s]')
ylabel('relative ground reactio force [-]')
legend('F_y','F_x')

%% Equation of motion for single support phase
    function output = EoM_single(~,q)
        
        omega = sqrt(k/m);
        output(1,1) = q(3);
        output(2,1) = q(4);
        output(3,1) = q(1)*omega^2*(l0/sqrt(q(1)^2 + q(2)^2) - 1);
        output(4,1) = q(2)*omega^2*(l0/sqrt(q(1)^2 + q(2)^2) - 1) - g;
    end
%% Equation of motion for double support phase
    function output = EoM_double(~,q)
        
        
        output(1,1) = q(3);
        output(2,1) = q(4);
        output(3,1) = -k*(q(1) - l0*q(1)*(q(1)^2 + q(2)^2)^-0.5 + (q(1) - x1) - l0*(q(1) - x1)*((q(1) - x1)^2 + q(2)^2)^-0.5)/m;
        output(4,1) = -k*(2*q(2) - l0*q(2)*(q(1)^2 + q(2)^2)^-0.5 - l0*q(2)*((q(1) - x1)^2 + q(2)^2)^-0.5)/m - g;
    end
%% End of Step conditions
    function [impact,terminate,direction] = apex(~,q)
        impact = q(4);
        terminate = 1;
        direction = -1;
    end
%% Push-off Conditions
    function [impact,terminate,direction] = single_stance(~,q)
        impact = sqrt((q(1)^2 + q(2)^2)) - l0;
        terminate = 1;
        direction = 1;
    end
%% Touchdown Conditions
    function [impact,terminate,direction] = double_stance(~,q)
        
        impact = q(2) - l0*sin(alpha0);
        terminate = 1;
        direction = -1;
    end
end